/**
 * PenguinAttackAI.java
 *
 * Description: A Penguin Attack AI.
 *
 * Copyright (c) 2011, Jason Buck
 * 
 * Distributed under the BSD-new license. For details see the BSD_LICENSE file 
 * that should have been included with this distribution. If the source you 
 * acquired this distribution from incorrectly removed this file, the license 
 * may be viewed at http://www.opensource.org/licenses/bsd-license.php.
 */
import java.awt.Point;

public class PenguinAttackAI 
{
	// The network client.
	private PenguinAttackTCPClient client;
	// The board: [0][0] is the top-left block and [11][5] is the bottom right
	private PenguinAttackBlock[][] board;
	// The (x, y) coordinate of the left side of the cursor. Valid x coords are
	// 1 - 11, valid y coors are 0 - 4.
	private Point cursorPosition;
	// The number of blocks of a given color on a given row first index is the
	// row in question, second index is the color code.
	private int blockCount[][];
	// How much stoppage time we have remaining.
	private int timeUntilScroll;
	// How long until the queued line becomes active.
	private int timeUntilNewLine;

	/**
	 * PenguinAttackAI constructor
	 *
	 * @param client The client we will be sending moves to.
	 */
	public PenguinAttackAI(PenguinAttackTCPClient client) 
	{
		this.client = client;
		this.board = new PenguinAttackBlock[12][6];
		this.blockCount = new int[12][7];
		this.cursorPosition = new Point(2, 2);
	}

	/*
	 * This is the method that is called once the AI object of this class is
	 * instantiated.  It is the main event loop for the AI.
	 */
	public void playPenguinAttack() 
	{
		this.client.resetGameOver();
		// The event loop.
		this.updateBoardState();
		while (!this.client.gameOver() && client.isConnected())
		{			
			while(!this.client.gameStarted())
			{
				try 
				{
					Thread.sleep(5);
				}
				catch(Exception e) {}
			}
			
			this.raiseStackToSafeHeight();
			this.breakGarbage();
			this.raiseStackToSafeHeight();
			this.levelOffStack();
			this.raiseStackToSafeHeight();
			this.breakGarbage();
			this.raiseStackToSafeHeight();
			this.makeCombosAndChains();
		}

		if(this.client.isConnected())
		{
			if(this.client.aiIsWinner())
				System.out.println("The AI won.");
			else
				System.out.println("The AI lost.");
		}
	}

	/*
	 * Counts how many normal blocks are on the board.
	 * NOTE:  Didn't end up using this.
	 */
	private int countBlocks()
	{
		int blockCount = 0;
		this.updateBoardState();
		for(int row = 0; row < this.blockCount.length; row++)
			for(int color = 0; color < this.blockCount[row].length; color++)
				blockCount += this.blockCount[row][color];
		return blockCount;
	}

	/*
	 * Updates our timing information to match (roughly since server and client
	 * are not synchronized) the server's.  
	 * NOTE:  Didn't end up using this.
	 */
	private void updateTimer() 
	{
		// Request the current timer
		this.client.sendCommand("w");

		while (!this.client.receivedNewTimerState()) 
		{
			if(this.client.gameOver() || !this.client.isConnected() || !this.client.gameStarted())
				return;
			
			try 
			{
				Thread.sleep(1);
			} 
			catch (Exception e) {}			
		}

		String newTimer = this.client.getTimer();
		newTimer = newTimer.substring(1, newTimer.length());
		this.timeUntilScroll = Integer.parseInt(newTimer.substring(0, newTimer.indexOf(' ')));
		this.timeUntilNewLine = Integer.parseInt(newTimer.substring(newTimer.indexOf(' ') + 1, newTimer.length() - 1));
	}

	/*
	 * Gets the current board state from the server and re-initializes our 2D
	 * array to match.
	 */
	private void updateBoardState() 
	{
		// Request the current board state
		this.client.sendCommand("b");

		// Reset the blockCount array;
		for (int row = 0; row < this.blockCount.length; row++)
			for (int col = 0; col < this.blockCount[row].length; col++)
				this.blockCount[row][col] = 0;

		while (!this.client.receivedNewBoardState()) 
		{
			if(this.client.gameOver() || !this.client.isConnected() || !this.client.gameStarted())
				return;
			
			try 
			{
				Thread.sleep(1);
			} 
			catch (Exception e) {}			
		}

		String[] boardStrings = this.client.getBoardState();

		try 
		{
			// Extract only the lines visible to the player
			for (int stringRow = boardStrings.length - 13, row = 0; stringRow < (boardStrings.length - 1); stringRow++, row++) 
			{
				// Populate the board array with PA blocks (each block is
				// described by a 4-character string).
				for (int col = 0; col < board[row].length; col++) 
				{
					board[row][col] = new PenguinAttackBlock(boardStrings[stringRow].substring(4*col, 4*col + 4));
					if(this.board[row][col].isNormalBlock())
						blockCount[row][board[row][col].getColor()]++;
				}
			}
		} 
		catch (Exception e) 
		{
			System.out.println("Caught Exception: " + e.getCause());
			e.printStackTrace();
		}
	}

	/*
	 * Re-synchronizes where we think the cursor is with where the server says
	 * it is.
	 */
	private void updateCursorPosition()
	{		
		// Request the current cursor position
		this.client.sendCommand("c");
		
		while (!this.client.receivedNewCursorPosition()) 
		{
			if(client.gameOver() || !this.client.isConnected() || !this.client.gameStarted())
				return;
			
			try 
			{
				Thread.sleep(1);
			} 
			catch (Exception e) {}
		}

		String cursorPositionString = client.getCursorPosition();

		cursorPosition.x = Integer.valueOf(String.valueOf(cursorPositionString.charAt(1)));
		cursorPositionString = cursorPositionString.substring(3, cursorPositionString.length() - 1);
		cursorPosition.y = Integer.valueOf(cursorPositionString);
	}

	/*
	 * Generates the move sequence to move a block from fromPosition to
	 * toPosition.  Is sometimes thwarted by accidental combos.
	 */
	private void moveBlock(Point fromPosition, int toPosition) 
	{
		if (fromPosition.x == toPosition || fromPosition.x < 0 || fromPosition.x > board[0].length - 1 || fromPosition.y < 0 || fromPosition.y > board.length - 1 || toPosition < 0 || toPosition > board[0].length - 1)
			return;
		
		this.updateBoardState();

		int i, j;
		// If things are breaking along the path we want to take, don't move
		if (fromPosition.x < toPosition)
		{
			i = fromPosition.x;
			j = toPosition;
		}
		else
		{
			i = toPosition;
			j = fromPosition.x;
		}
		for(; i <= j; i++, j--)
		{
			if(i < board[fromPosition.y].length && !board[fromPosition.y][i].blockIsIdle())
				return;
			if(j > 0 && !board[fromPosition.y][j].blockIsIdle())
				return;
		}

		String moveSequence = "";
		this.updateCursorPosition();

		// Generate down sequence (cursor is above block to be moved)
		while (cursorPosition.y < fromPosition.y) 
		{
			moveSequence += "d";
			cursorPosition.y++;
		}

		// Generate up sequence (cursor is below the block to be moved)
		while (cursorPosition.y > fromPosition.y) 
		{
			moveSequence += "u";
			cursorPosition.y--;
		}

		// Generate horizontal move and swap sequence
		if (fromPosition.x < toPosition) 
		{
			while (cursorPosition.x < fromPosition.x) 
			{
				moveSequence += "r";
				cursorPosition.x++;
			}
			while (cursorPosition.x > fromPosition.x) 
			{
				moveSequence += "l";
				cursorPosition.x--;
			}
			while (fromPosition.x != toPosition - 1) 
			{
				moveSequence += "sr";
				fromPosition.x++;
				cursorPosition.x++;
			}
		} 
		else 
		{
			while (cursorPosition.x + 1 < fromPosition.x) 
			{
				moveSequence += "r";
				cursorPosition.x++;
			}
			while (cursorPosition.x + 1 > fromPosition.x) 
			{
				moveSequence += "l";
				cursorPosition.x--;
			}
			while (fromPosition.x != toPosition + 1) 
			{
				moveSequence += "sl";
				fromPosition.x--;
				cursorPosition.x--;
			}
		}

		// Add final swap (so that we don't have an extra l or r on the end)
		moveSequence += "s";

		// Send the move sequence
		this.client.sendMoveSequence(moveSequence);
	}

	/*
	 * Checks to see if there is any garbage on the board.
	 */
	private boolean thereIsGarbage()
	{
		this.updateBoardState();
		for(int row = 0; row < this.board.length; row++)
		{
			for (int col = 0; col < this.board[row].length; col++) 
			{
				if (this.board[row][col].isGarbageBlock()) 
					return true;
			}
		}

		return false;
	}

	/*
	 * Returns the number of the row containing the highest normal block.
	 */
	private int findTopBlock()
	{
		this.updateBoardState();
		// Find the y-coordinate of the highest block
		for (int row = 0; row < this.board.length; row++) 
		{
			for (int col = 0; col < this.board[row].length; col++) 
			{
				if (this.board[row][col].isNormalBlock())
					return row;
			}
		}
		
		// This could happen in the extremely unlikely event that we have used 
		// all of our blocks
		return -1;
	}

	/*
	 * Returns the number of the row containing the highest non-empty space.
	 */
	private int findStackHeight()
	{
		this.updateBoardState();
		// Find the height of the stack (including garbage).
		for (int row = 0; row < this.board.length; row++) 
		{
			for (int col = 0; col < this.board[row].length; col++) 
			{
				if (this.board[row][col].isEmptySpace()) 
					continue;
				else 
					return row;
			}
		}

		return this.board.length - 1;
	}

	/* 
	 * Utility function to calculate grid distance 
	 */
	private int gridDistance(int x1, int y1, int x2, int y2)
	{
		return ((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
	}

	/*
	 * Levels off the stack.
	 */
	private void levelOffStack()
	{
		int row, col;
		Point emptySpace = new Point(0,0);
		Point fillerBlock = new Point(0,0);
		boolean foundEmptySpace, foundFillerBlock, leveled = false;
		
		while(leveled == false)
		{
			if(this.client.gameOver() || !this.client.isConnected() || !this.client.gameStarted())
				return;
			
			this.updateBoardState();

			// Start at the bottom and move up.
			for(foundEmptySpace = false, foundFillerBlock = false, row = this.board.length - 1; row > 1; row--)
			{
				// Search for an empty space.
				if(!foundEmptySpace) 
				{
					for(col = 0; col < this.board[row].length; col++)
					{
						// Stop at the first empty space
						if(this.board[row][col].isEmptySpace()) 
						{
							emptySpace.x = col;
							emptySpace.y = row;
							foundEmptySpace = true;
							break;
						}
					}
				}
				else
				{
					// We found an empty space; find a block in the row immediately above it.

					for(col = 0; col < this.board[row].length; col++)
					{
						if(this.board[row][col].isGarbageBlock())
						{
							if(row >= this.findTopBlock())
								leveled = true;
							else
								foundEmptySpace = false;

							break;
						}						
						if(this.board[row][col].isNormalBlock()) 
						{
							if(!foundFillerBlock) {
								fillerBlock.x = col;
								fillerBlock.y = row;
								foundFillerBlock = true;
							}
							else
							{
								// Make sure the block is the closest one to the space.
								if(gridDistance(emptySpace.x, emptySpace.y, col, row) <
										gridDistance(emptySpace.x, emptySpace.y, fillerBlock.x, fillerBlock.y)) 
								{
									fillerBlock.x = col;
									fillerBlock.y = row;
								}
							}
						}
					}

					if(foundFillerBlock) 
						break;
				}
			}

			// If we couldn't find a filler block then the stack is leveled.
			if(!foundFillerBlock)
				leveled = true;
			else
			{
				// Make sure the empty space is the closest empty space below the block.
				if(emptySpace.x < fillerBlock.x) {
					for(col = emptySpace.x; col < fillerBlock.x; col++)
						if(this.board[emptySpace.y][col].isEmptySpace())
							emptySpace.x = col;
				}
				else
				{
					for(col = emptySpace.x; col > fillerBlock.x; col--)
						if(this.board[emptySpace.y][col].isEmptySpace())
							emptySpace.x = col;
				}
				this.moveBlock(fillerBlock, emptySpace.x);
			}
		}
	}

	/*
	 * If the stack is below the maximum "safe" height, this method generates
	 * and sends the move sequence to raise the garbage to that maximum "safe"
	 * height.  It is sometimes thwarted by accidental combos.
	 */
	private void raiseStackToSafeHeight() 
	{
		this.updateBoardState();
		int topRow = this.findStackHeight();

		this.updateCursorPosition();
		
		// Raise the stack to the maximum "safe" height.
		if (topRow > 1) 
		{
			String raiseSequence = "";
			while (topRow != 1) 
			{
				raiseSequence += "m";
				topRow--;
				this.cursorPosition.y--;
			}
			this.client.sendMoveSequence(raiseSequence);
		}
	}

	/*
	 * Searches the board for garbage, and then searches the possible ways to
	 * break that garbage, returning once it finds one, or it has exhausted
	 * possible ways to break the garbage.
	 */
	private void breakGarbage()
	{
		boolean brokeGarbage = false;
		this.updateBoardState();

		find_garbage:
			for (int row = this.board.length - 2; row >= 0; row--) 
			{
				for (int col = 0; col < this.board[row].length; col++) 
				{
					if(!this.board[row][col].isGarbageBlock() || !this.board[row][col].blockIsIdle())
						continue;

					if (this.board[row + 1][col].isNormalBlock()) 
					{
						brokeGarbage = breakGarbageWithVertCombo(row + 1, col, 1);

						if(!brokeGarbage)
							brokeGarbage = breakGarbageWithHorizCombo(row + 1, col);

						if (brokeGarbage)
							break find_garbage;
					}

					if (row - 1 > 0 && this.board[row - 1][col].isNormalBlock()) 
					{
						brokeGarbage = breakGarbageWithVertCombo(row - 1, col, -1);

						if(!brokeGarbage)
							brokeGarbage = breakGarbageWithHorizCombo(row - 1, col);

						if (brokeGarbage)
							break find_garbage;
					}
				}
			}
	}

	/*
	 * Given a block that is touching garbage, this method attempts to find a 
	 * vertical combo that will break that garbage.  Returns true if it is
	 * successful, otherwise false.
	 */
	private boolean breakGarbageWithVertCombo(int i, int j, int direction)
	{
		if (i + 2*direction > 11 || i + 2 * direction < 1)
			return false;

		if(this.blockCount[i + direction][this.board[i][j].getColor()] < 1 || this.blockCount[i + 2*direction][this.board[i][j].getColor()] < 1)
			return false;

		int second = -1, third = -1;

		for (int k = 0; k < board[i].length; k++) 
		{
			if (second == -1 && this.board[i + direction][k].isNormalBlock() 
					&& this.board[i + direction][k].getColor() == this.board[i][j].getColor()) 
				second = k;
			if (third == -1 && this.board[i + 2*direction][k].isNormalBlock() && this.board[i + 2*direction][k].getColor() == this.board[i][j].getColor()) 
				third = k;
			if (second != -1 && third != -1)
				break;
		}

		this.moveBlock(new Point(third, i + 2*direction), j);
		this.moveBlock(new Point(second, i + direction), j);
		return true;
	}

	/*
	 * Given a block that is touching garbage, this method attempts to find a 
	 * horizontal combo that will break that garbage.  Returns true if it is
	 * successful, otherwise false.
	 */
	private boolean breakGarbageWithHorizCombo(int i, int j)
	{
		if(this.blockCount[i][this.board[i][j].getColor()] < 3)
			return false;

		int second = -1, third = -1;
		for (int k = 0; k < this.board[i].length; k++) 
		{
			if (k == j)
				continue;

			if (this.board[i][k].isNormalBlock() && this.board[i][k].getColor() == this.board[i][j].getColor())
			{
				if(second == -1)
					second = k;
				else 
				{
					third = k;
					break;
				}
			}
		}

		if (second < j && third < j)
		{
			this.moveBlock(new Point(third, i), j - 1);
			this.moveBlock(new Point(second, i), j - 2);
		}
		else if (second > j && third > j)
		{
			this.moveBlock(new Point(second, i), j + 1);
			this.moveBlock(new Point(third, i), j + 2);
		}
		else if (second > j && third < j)
		{
			this.moveBlock(new Point(second, i), j + 1);
			this.moveBlock(new Point(third, i), j - 1);
		}
		else 
		{
			this.moveBlock(new Point(third, i), j + 1);
			this.moveBlock(new Point(second, i), j - 1);
		}

		return true;
	}

	/*
	 * While this has had code in it before, none of it worked well, so this is
	 * really just a placeholder for future work.
	 */
	private void makeChains()
	{
	
	}
	
	/*
	 * Start by making a combo, then attempt to turn it into a chain. 
	 */
	private void makeCombosAndChains() 
	{
		boolean madeVertCombo = false;
		boolean madeHorizCombo =  false;
		boolean lowest = true;

		this.updateBoardState();

		if(this.thereIsGarbage())
			lowest = false;

		madeHorizCombo = this.makeExtremeHorizCombo(lowest);

		if (!madeHorizCombo) 
		{
			madeVertCombo = this.makeExtremeVertCombo(lowest);
		}

		if (madeVertCombo || madeHorizCombo) 
		{
			makeChains();
		}
		else 
		{
			System.out.println("Can't find a combo to make!");
		}
	}

	
	
	/*
	 * Makes a vertical combo as close to the bottom of the stack as possible
	 * if lowest is true, otherwise makes the vertical combo as close to the
	 * top of the stack as possible.
	 */
	private boolean makeExtremeVertCombo(boolean lowest)
	{
		this.updateBoardState();
		int row, incrementAmt, lowerLimit, upperLimit, comboColor = -1;

		incrementAmt = lowest ? -1 : 1;
		lowerLimit = lowest ? 2 : 0; // going up we don't want start a combo above row 3, doing down we don't care.
		upperLimit = lowest ? this.board.length : this.board.length - 2; // going up we don't care, going down don't want to start above row 10.

		for (row = lowest ? this.board.length - 1 : 1; row > lowerLimit && row < upperLimit; row += incrementAmt) 
		{
			comboColor = this.findVertComboColorOn(row, incrementAmt);
			if(comboColor != -1)
				break;
		}

		if(comboColor == -1)
			return false;

		int first = -1, second = -1, third = -1; // Col's of the blocks of the combo
		for (int col = 0; col < this.board[row].length; col++)
		{
			if(first == -1 && this.board[row][col].isNormalBlock() && this.board[row][col].getColor() == comboColor)
				first = col;
			if(second == -1 && this.board[row + incrementAmt][col].isNormalBlock() && this.board[row + incrementAmt][col].getColor() == comboColor)
				second = col;
			if(third == -1 && this.board[row + 2*incrementAmt][col].isNormalBlock() && this.board[row + 2*incrementAmt][col].getColor() == comboColor)
				third = col;
			if(first != -1 && second != -1 && third != -1)
				break;
		}

		this.moveBlock(new Point(second, row + incrementAmt), first);
		this.moveBlock(new Point(third, row + 2*incrementAmt), first);
		return true;
	}

	/*
	 * Finds a vertical combo starting on a given row and going down if
	 * incrementAmt is 1 and going up if incrementAmt is -1.  
	 * 
	 */
	private int findVertComboColorOn(int row, int incrementAmt)
	{
		for(int color = 0; color < this.blockCount[row].length; color++)
		{
			if(this.blockCount[row][color] > 0 && this.blockCount[row + incrementAmt][color] > 0 && this.blockCount[row + 2*incrementAmt][color] > 0)
				return color;
		}

		return -1;
	}

	/*
	 * Makes a horizontal combo as close to the bottom of the stack as possible
	 * if lowest is true, otherwise makes the horizontal combo as close to the
	 * top of the stack as possible.
	 */
	private boolean makeExtremeHorizCombo(boolean lowest)
	{
		this.updateBoardState();
		int row, incrementAmt, comboColor = -1;

		incrementAmt = lowest ? -1 : 1;

		// Find the lowest line that contains a horizontal potential combo
		for (row = lowest ? this.board.length - 1 : 1; row > 0 && row < board.length; row += incrementAmt) 
		{
			comboColor = this.findHorizComboColorOn(row);
			if(comboColor != -1)
				break;
		}

		if(comboColor == -1)
			return false;

		int first = -1, second = -1, third = -1;  // Col's of the blocks of the combo
		for (int col = 0; col < this.board[row].length; col++) 
		{
			if (first == -1 && this.board[row][col].isNormalBlock() && this.board[row][col].getColor() == comboColor) 
				first = col;
			else if (second == -1 && this.board[row][col].isNormalBlock() && this.board[row][col].getColor() == comboColor)
				second = col;
			else if (this.board[row][col].isNormalBlock() && this.board[row][col].getColor() == comboColor){
				third = col;
				break;
			}
		}

		if( first != -1 && second != -1 && third != -1 )
		{
			this.moveBlock(new Point(second, row), first + 1);
			this.moveBlock(new Point(third, row), first + 2);
			return true;
		}

		return false;
	}

	/*
	 * Finds a horizontal combo on a given row and returns its color.  Returns 
	 * -1 if a horizontal combo cannot be made on that row.
	 */
	private int findHorizComboColorOn(int row)
	{
		for(int color = 0; color < this.blockCount[row].length; color++)
			if (blockCount[row][color] >= 3)
				return color;

		return -1;
	}
}
